/*---------------------------------------------------------------------------*\

    CWINKMDL.C      Version 1.0                                      2000 AVM

    This file contains the source of the operating system specific
    CAPI functions, here for Windows - kernel mode driver level.
    According to CAPI 2.0 spec part II, chapter 8.15.

\*---------------------------------------------------------------------------*/
#ifdef WDM
#   include <wdm.h>
#else
#   include <ntddk.h>
#endif

#include "cwinkmdl.h"
#include "capi20.h"
#include "cerrfkt.h"
#include "ckmdev.h"
#include "ckmdldef.h"
#include "kassert.h"


/*--------------------------------------------------------------------------*\
\*--------------------------------------------------------------------------*/
CAPI_REGISTER_ERROR CAPI20_REGISTER (unsigned MsgBufSize,
                                     unsigned MaxB3Connection,
                                     unsigned MaxB3Blks,
                                     unsigned MaxSizeB3,
                                     void **hCapi) {

    NTSTATUS               status;
    CAPI_REGISTER_ERROR error;
    capi_register_params   RegisterParams;

    assert (hCapi != 0);
    assert (KeGetCurrentIrql () == PASSIVE_LEVEL);

    if (DeviceObjectOpen(hCapi) != STATUS_SUCCESS) {
        return CapiRegNotInstalled;
    }

    assert (*hCapi != 0);

    RegisterParams.MessageBufferSize     = MsgBufSize;
    RegisterParams.maxLogicalConnections = MaxB3Connection;
    RegisterParams.maxBDataBlocks        = MaxB3Blks;
    RegisterParams.maxBDataLen           = MaxSizeB3;

    status = DeviceObjectRequest (*hCapi,
                 (unsigned long) IOCTL_CAPI_REGISTER,
                 sizeof (capi_register_params),
                 0,
                 &RegisterParams);

    error = NtToCapiRegisterError (status);

    if (error != CapiRegNoError) {
        DeviceObjectClose(*hCapi);
        *hCapi = 0;
    }

    return error;
}


/*--------------------------------------------------------------------------*\
\*--------------------------------------------------------------------------*/
MESSAGE_EXCHANGE_ERROR CAPI20_RELEASE (void *hCapi) {

    NTSTATUS status;

    assert (KeGetCurrentIrql () == PASSIVE_LEVEL);

    status = DeviceObjectRequest (hCapi,
                 (unsigned long) IOCTL_CAPI_RELEASE,
                 0,
                 0,
                 0);

    return NtToMessageExchangeError (status);
}


/*--------------------------------------------------------------------------*\
\*--------------------------------------------------------------------------*/
unsigned CAPI20_ISINSTALLED (void) {

    void *hCapi;

    assert (KeGetCurrentIrql () == PASSIVE_LEVEL);

    if (DeviceObjectOpen(&hCapi) != STATUS_SUCCESS) {
        return 0;
    }

    DeviceObjectClose(hCapi);
    return 1;
}


/*--------------------------------------------------------------------------*\
\*--------------------------------------------------------------------------*/
MESSAGE_EXCHANGE_ERROR CAPI20_PUT_MESSAGE (void *hCapi, unsigned char *Msg) {

    NTSTATUS            status;
    capi_message_params PutMsgParams;
    CapiMessage         *pCapiMsg;

    assert (Msg != 0);
    assert (KeGetCurrentIrql () == PASSIVE_LEVEL);

    pCapiMsg = (CapiMessage *) Msg;

    PutMsgParams.message = Msg;
    PutMsgParams.data    = (  (pCapiMsg->Command == CAPI_DATA_B3)
                           && (pCapiMsg->SubCommand == CAPI_REQ)) ?
                           pCapiMsg->DataB3.Data : 0;

    status = DeviceObjectRequest (hCapi,
                 (unsigned long) IOCTL_CAPI_PUT_MESSAGE,
                 sizeof (capi_message_params),
                 sizeof (capi_message_params),
                 &PutMsgParams);

    return NtToMessageExchangeError (status);
}


/*--------------------------------------------------------------------------*\
\*--------------------------------------------------------------------------*/
MESSAGE_EXCHANGE_ERROR CAPI20_GET_MESSAGE (void *hCapi, unsigned char **ReturnMessage) {

    CapiMessage            *pCapiMsg;
    NTSTATUS               status;
    MESSAGE_EXCHANGE_ERROR error;
    capi_message_params    GetMsgParams;

    assert (ReturnMessage != 0);
    assert (KeGetCurrentIrql () == PASSIVE_LEVEL);

    status = DeviceObjectRequest (hCapi,
                 (unsigned long) IOCTL_CAPI_GET_MESSAGE,
                 sizeof (capi_message_params),
                 sizeof (capi_message_params),
                 &GetMsgParams);

    error = NtToMessageExchangeError (status);

    if (error == CapiNoError) {
        assert (GetMsgParams.message != 0);
        *ReturnMessage = GetMsgParams.message;
        pCapiMsg       = GetMsgParams.message;

        if (   (pCapiMsg->Command == CAPI_DATA_B3)
            && (pCapiMsg->SubCommand == CAPI_IND)) {
            pCapiMsg->DataB3.Data = GetMsgParams.data;
        }
    }

    return error;
}


/*--------------------------------------------------------------------------*\
\*--------------------------------------------------------------------------*/
MESSAGE_EXCHANGE_ERROR CAPI20_GET_PROFILE (unsigned Controller, CAPI_GET_PROFILE_STRUCT *profile) {

    NTSTATUS               status;
    void                   *hCapi;
    MESSAGE_EXCHANGE_ERROR error;

    assert (profile != 0);
    assert (KeGetCurrentIrql () == PASSIVE_LEVEL);

    status = DeviceObjectOpen(&hCapi);

    if (status != STATUS_SUCCESS) {
        return NtToMessageExchangeError (status);
    }

    memset (profile, 0, sizeof (CAPI_GET_PROFILE_STRUCT));
    *(unsigned *)profile = Controller;

    status = DeviceObjectRequest (hCapi,
                 (unsigned long) IOCTL_CAPI_GET_PROFILE,
                 sizeof (unsigned),
                 sizeof (CAPI_GET_PROFILE_STRUCT),
                 profile);

    error = NtToMessageExchangeError (status);
    DeviceObjectClose(hCapi);

    return error;
}


/*--------------------------------------------------------------------------*\
\*--------------------------------------------------------------------------*/
MESSAGE_EXCHANGE_ERROR CAPI20_GET_VERSION (unsigned Controller, capi_version_params *version) {


    NTSTATUS               status;
    void                   *hCapi;
    MESSAGE_EXCHANGE_ERROR error;

    assert (version != 0);
    assert (KeGetCurrentIrql () == PASSIVE_LEVEL);

    status = DeviceObjectOpen(&hCapi);

    if (status != STATUS_SUCCESS) {
        return NtToMessageExchangeError (status);
    }

    memset (version, 0, sizeof (capi_version_params));
    *(unsigned *)version = Controller;

    status = DeviceObjectRequest (hCapi,
                 (unsigned long) IOCTL_CAPI_GET_VERSION,
                 sizeof (unsigned),
                 sizeof (capi_version_params),
                 version);

    error = NtToMessageExchangeError (status);
    DeviceObjectClose(hCapi);

    return error;
}


/*--------------------------------------------------------------------------*\
\*--------------------------------------------------------------------------*/
MESSAGE_EXCHANGE_ERROR CAPI20_GET_SERIAL_NUMBER (unsigned Controller, ptr_capi_serial_number_buf sernum) {

    NTSTATUS               status;
    void                   *hCapi;
    MESSAGE_EXCHANGE_ERROR error;

    assert (sernum != 0);
    assert (KeGetCurrentIrql () == PASSIVE_LEVEL);

    status = DeviceObjectOpen(&hCapi);

    if (status != STATUS_SUCCESS) {
        return NtToMessageExchangeError (status);
    }

    memset (sernum, 0, sizeof (capi_serial_number_buf));
    *(unsigned *)sernum = Controller;

    status = DeviceObjectRequest (hCapi,
                 (unsigned long) IOCTL_CAPI_GET_SERIAL,
                 sizeof (unsigned),
                 sizeof (capi_serial_number_buf),
                 sernum);

    error = NtToMessageExchangeError (status);
    DeviceObjectClose(hCapi);

    return error;
}


/*--------------------------------------------------------------------------*\
\*--------------------------------------------------------------------------*/
MESSAGE_EXCHANGE_ERROR CAPI20_GET_MANUFACTURER (unsigned Controller, ptr_capi_manufacturer_buf manubuf) {

    NTSTATUS               status;
    void                   *hCapi;
    MESSAGE_EXCHANGE_ERROR error;

    assert (manubuf != 0);
    assert (KeGetCurrentIrql () == PASSIVE_LEVEL);

    status = DeviceObjectOpen(&hCapi);

    if (status != STATUS_SUCCESS) {
        return NtToMessageExchangeError (status);
    }

    memset (manubuf, 0, sizeof(capi_manufacturer_buf));
    *(unsigned *)manubuf = Controller;

    status = DeviceObjectRequest (hCapi,
                 (unsigned long) IOCTL_CAPI_GET_MANUFACTURER,
                 sizeof (unsigned),
                 sizeof(capi_manufacturer_buf),
                 manubuf);

    error = NtToMessageExchangeError (status);
    DeviceObjectClose(hCapi);

    return error;
}


/*---------------------------------------------------------------------------*\
\*---------------------------------------------------------------------------*/
