/* 
 * Copyright 2015 by AVM GmbH <info@avm.de>
 *
 * This software contains free software; you can redistribute it and/or modify 
 * it under the terms of the GNU General Public License ("License") as 
 * published by the Free Software Foundation  (version 3 of the License). 
 * This software is distributed in the hope that it will be useful, but 
 * WITHOUT ANY WARRANTY; without even the implied warranty of 
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the copy of the 
 * License you received along with this software for more details.
 */

package de.avm.fundamentals.fragments;

import android.graphics.drawable.GradientDrawable;
import android.os.Bundle;
import android.view.View;
import android.widget.ImageView;
import android.widget.RelativeLayout;
import android.widget.TextView;
import com.squareup.otto.Subscribe;
import de.avm.fundamentals.R;
import de.avm.fundamentals.events.OnIntroPageSelectedEvent;
import de.avm.fundamentals.utils.EventBusProvider;

public class IntroFragment extends BaseFragment {

    public static final String BUNDLE_TITLE_RES_ID      = "title_res_id";
    public static final String BUNDLE_MESSAGE_RES_ID    = "text_res_id";
    public static final String BUNDLE_IMAGE_RES_ID      = "image_res_id";
    public static final String BUNDLE_COLOR_TO          = "color_to";
    public static final String BUNDLE_COLOR_FROM        = "color_from";
    public static final String BUNDLE_POSITION          = "position";

    private int mTitleResId;
    private int mMessageResId;
    private int mImageResId;
    private int mColorFrom;
    private int mColorTo;
    private int mPosition;
    private ImageView mImage;
    private TextView mMessage;
    private TextView mTitle;
    private boolean mAnimate;

    @Override
    public void onCreate(Bundle savedInstanceState) {
        super.onCreate(savedInstanceState);
        applyArguments();
    }

    private void applyArguments() {
        final Bundle arguments = getArguments();
        if(arguments != null) {
            mPosition = arguments.getInt(BUNDLE_POSITION);
            mImageResId = arguments.getInt(BUNDLE_IMAGE_RES_ID);
            mMessageResId = arguments.getInt(BUNDLE_MESSAGE_RES_ID);
            mTitleResId = arguments.getInt(BUNDLE_TITLE_RES_ID);
            mColorFrom = getResources().getColor(arguments.getInt(BUNDLE_COLOR_FROM));
            mColorTo = getResources().getColor(arguments.getInt(BUNDLE_COLOR_TO));
        }
    }

    @Override
    public int getFragmentLayoutResId() {
        return R.layout.intro_fragment;
    }

    @Override
    public void initLayout(View view, Bundle savedInstanceState) {
        RelativeLayout container = (RelativeLayout) view.findViewById(R.id.intro_container);
        container.setBackgroundDrawable(getBackground());
        mImage = (ImageView) view.findViewById(R.id.intro_image);
        mImage.setImageResource(mImageResId);

        mTitle = (TextView) view.findViewById(R.id.intro_title);
        mTitle.setText(mTitleResId);
        mMessage = (TextView) view.findViewById(R.id.intro_message);
        mMessage.setText(mMessageResId);
        resetAnimation(OnIntroPageSelectedEvent.Direction.TO_RIGHT);
        mAnimate = (mPosition == 0);
    }

    @Override
    public void onResume() {
        super.onResume();
        EventBusProvider.getInstance().register(this);
        if(mAnimate) {
            animateIntroIn();
            mAnimate = false;
        }
    }

    @Override
    public void onPause() {
        super.onPause();
        EventBusProvider.getInstance().unregister(this);
    }

    public GradientDrawable getBackground(){
        GradientDrawable gd = new GradientDrawable(GradientDrawable.Orientation.RIGHT_LEFT,  new int[] {mColorTo, mColorFrom });
        gd.setCornerRadius(0f);
        return gd;
    }

    @Override
    public void setUserVisibleHint(boolean isVisibleToUser) {
        super.setUserVisibleHint(isVisibleToUser);
        if(!isVisibleToUser && mImage != null && mMessage != null){
            resetAnimation(OnIntroPageSelectedEvent.Direction.TO_RIGHT);
        }
    }

    private void resetAnimation(OnIntroPageSelectedEvent.Direction mDirection) {
        boolean isRight = mDirection.equals(OnIntroPageSelectedEvent.Direction.TO_RIGHT);
        mImage.setTranslationX(isRight ? 70 : -70);
        mMessage.setTranslationY(20);
        mTitle.setTranslationY(20);
        mImage.setAlpha(0f);
        mMessage.setAlpha(0f);
        mTitle.setAlpha(0f);

    }

    @Subscribe
    public void onPageSelected(OnIntroPageSelectedEvent event){
        if(mPosition == event.mPosition){
            resetAnimation(event.mDirection);
            animateIntroIn();
        }
    }

    private void animateIntroIn() {
        mImage.animate().translationX(0).alpha(1.0f).setDuration(500).start();
        mMessage.animate().translationY(0).alpha(1.0f).setDuration(300).setStartDelay(200).start();
        mTitle.animate().translationY(0).alpha(1.0f).setDuration(300).setStartDelay(200).start();
    }

    public void setAnimateIntroIn(boolean shouldAnimate){
        mAnimate = shouldAnimate;
    }
}
